<!DOCTYPE html>
<html lang="pl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <meta name="theme-color" content="#000000">
    <meta name="apple-mobile-web-app-capable" content="yes">
    <meta name="apple-mobile-web-app-status-bar-style" content="black-translucent">
    <title>Odtwarzacz MP3</title>
    <style>
        :root {
            --primary-color: #90EE90;
            --bg-color: #000000;
            --panel-bg: #333333;
            --panel-border: #444444;
            --progress-bg: #222222;
            --highlight-bg: #555555;
        }

        * {
            box-sizing: border-box;
            -webkit-tap-highlight-color: transparent;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            margin: 0;
            padding: 10px;
            background-color: var(--bg-color);
            color: var(--primary-color);
            display: flex;
            flex-direction: column;
            height: 100vh;
            user-select: none;
            overflow: hidden;
            touch-action: manipulation;
        }

        h1 {
            font-size: clamp(1.2rem, 5vw, 1.8rem);
            margin: 5px 0;
            padding-bottom: 8px;
            border-bottom: 1px solid var(--highlight-bg);
            text-align: center;
        }

        .container {
            display: flex;
            flex: 1;
            gap: 10px;
            min-height: 0;
            flex-direction: column;
        }

        @media (min-width: 768px) {
            .container {
                flex-direction: row;
            }
        }

        .explorer, .player {
            background-color: var(--panel-bg);
            border-radius: 8px;
            padding: 10px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.5);
            border: 1px solid var(--panel-border);
            overflow: hidden;
        }

        .explorer {
            flex: 1;
            min-height: 30vh;
        }

        @media (min-width: 768px) {
            .explorer {
                min-height: auto;
            }
        }

        .player {
            flex: 2;
            display: flex;
            flex-direction: column;
        }

        .controls {
            display: grid;
            grid-template-columns: repeat(5, 1fr);
            gap: 5px;
            margin-bottom: 10px;
            align-items: center;
        }

        @media (max-width: 480px) {
            .controls {
                grid-template-columns: repeat(3, 1fr);
            }
            .volume-control {
                grid-column: span 3;
            }
        }

        button {
            padding: 10px 5px;
            cursor: pointer;
            background-color: var(--highlight-bg);
            color: var(--primary-color);
            border: none;
            border-radius: 4px;
            transition: background-color 0.2s;
            font-size: clamp(0.8rem, 3vw, 1rem);
            min-height: 40px;
        }

        button:hover {
            background-color: #666666;
        }

        button:active {
            background-color: #777777;
        }

        .volume-control {
            display: flex;
            align-items: center;
            gap: 5px;
            grid-column: span 2;
        }

        @media (min-width: 768px) {
            .volume-control {
                grid-column: span 1;
            }
        }

        .progress-container {
            margin: 10px 0;
        }

        progress {
            width: 100%;
            height: 6px;
            border-radius: 3px;
            background-color: var(--progress-bg);
        }

        progress::-webkit-progress-bar {
            background-color: var(--progress-bg);
        }

        progress::-webkit-progress-value {
            background-color: var(--primary-color);
        }

        progress::-moz-progress-bar {
            background-color: var(--primary-color);
        }

        .time-display {
            display: flex;
            justify-content: space-between;
            font-size: clamp(0.7rem, 2vw, 0.8rem);
        }

        #fileList {
            list-style-type: none;
            padding: 0;
            margin: 0;
            height: calc(100% - 10px);
            overflow-y: auto;
            -webkit-overflow-scrolling: touch;
        }

        #fileList li {
            padding: 10px 8px;
            border-bottom: 1px solid var(--panel-border);
            cursor: pointer;
            display: flex;
            align-items: center;
            font-size: clamp(0.8rem, 2.5vw, 1rem);
        }

        #fileList li:hover {
            background-color: var(--panel-border);
        }

        #fileList li.folder {
            font-weight: bold;
        }

        #fileList li.file {
            padding-left: 25px;
        }

        #fileList li.playing {
            background-color: var(--highlight-bg);
        }

        #nowPlaying {
            margin: 10px 0;
            padding: 8px;
            background-color: var(--panel-border);
            border-radius: 4px;
            font-weight: bold;
            border-left: 4px solid var(--primary-color);
            font-size: clamp(0.8rem, 2.5vw, 1rem);
            text-align: center;
            overflow: hidden;
            text-overflow: ellipsis;
            white-space: nowrap;
        }

        .file-icon {
            margin-right: 8px;
            font-size: clamp(0.9rem, 3vw, 1.1rem);
            min-width: 20px;
            text-align: center;
        }

        .player-content {
            flex: 1;
            display: flex;
            flex-direction: column;
        }

        #oscilloscope {
            flex: 1;
            margin-top: 10px;
            background-color: var(--progress-bg);
            border-radius: 4px;
            border: 1px solid var(--highlight-bg);
            min-height: 80px;
        }

        input[type="range"] {
            -webkit-appearance: none;
            height: 6px;
            background: var(--progress-bg);
            border-radius: 3px;
            flex: 1;
        }

        input[type="range"]::-webkit-slider-thumb {
            -webkit-appearance: none;
            width: 15px;
            height: 15px;
            background: var(--primary-color);
            border-radius: 50%;
            cursor: pointer;
        }
    </style>
</head>
<body>
    <h1>Odtwarzacz MP3</h1>
    
    <div class="container">
        <div class="explorer">
            <ul id="fileList"></ul>
        </div>
        
        <div class="player">
            <div class="player-content">
                <audio id="audioPlayer"></audio>
                
                <div id="nowPlaying">Wybierz utwór z listy</div>
                
                <div class="progress-container">
                    <progress id="progressBar" value="0" max="100"></progress>
                    <div class="time-display">
                        <span id="currentTime">0:00</span>
                        <span id="duration">0:00</span>
                    </div>
                </div>
                
                <div class="controls">
                    <button id="prevBtn" title="Poprzedni utwór">◄◄</button>
                    <button id="playBtn" title="Odtwarzaj">►</button>
                    <button id="pauseBtn" title="Pauza">❚❚</button>
                    <button id="stopBtn" title="Stop">■</button>
                    <button id="nextBtn" title="Następny utwór">►►</button>
                    
                    <div class="volume-control">
                        <span title="Głośność">🔊</span>
                        <input type="range" id="volumeControl" min="0" max="1" step="0.1" value="1">
                    </div>
                </div>
                
                <canvas id="oscilloscope"></canvas>
            </div>
        </div>
    </div>

    <script>
        // Service Worker dla odtwarzania w tle
        if ('serviceWorker' in navigator) {
            navigator.serviceWorker.register('sw.js').then(registration => {
                console.log('ServiceWorker registration successful');
            }).catch(err => {
                console.log('ServiceWorker registration failed: ', err);
            });
        }

        document.addEventListener('DOMContentLoaded', function() {
            // Elementy DOM
            const audioPlayer = document.getElementById('audioPlayer');
            const playBtn = document.getElementById('playBtn');
            const pauseBtn = document.getElementById('pauseBtn');
            const stopBtn = document.getElementById('stopBtn');
            const prevBtn = document.getElementById('prevBtn');
            const nextBtn = document.getElementById('nextBtn');
            const fileList = document.getElementById('fileList');
            const nowPlaying = document.getElementById('nowPlaying');
            const progressBar = document.getElementById('progressBar');
            const currentTimeDisplay = document.getElementById('currentTime');
            const durationDisplay = document.getElementById('duration');
            const volumeControl = document.getElementById('volumeControl');
            const canvas = document.getElementById('oscilloscope');
            const ctx = canvas.getContext('2d');

            // Zmienne stanu
            let audioContext, analyser, source;
            let isAudioContextInitialized = false;
            let allSongs = [];
            let currentPlaylist = [];
            let currentSongIndex = -1;
            const baseDir = 'mp3';

            // Inicjalizacja kontekstu audio
            function initAudioContext() {
                if (!isAudioContextInitialized) {
                    audioContext = new (window.AudioContext || window.webkitAudioContext)();
                    analyser = audioContext.createAnalyser();
                    source = audioContext.createMediaElementSource(audioPlayer);
                    source.connect(analyser);
                    analyser.connect(audioContext.destination);
                    analyser.fftSize = 2048;
                    isAudioContextInitialized = true;
                }
            }

            // Media Session API dla sterowania z ekranu blokady
            if ('mediaSession' in navigator) {
                navigator.mediaSession.setActionHandler('play', () => {
                    audioPlayer.play();
                });
                navigator.mediaSession.setActionHandler('pause', () => {
                    audioPlayer.pause();
                });
                navigator.mediaSession.setActionHandler('previoustrack', () => {
                    prevBtn.click();
                });
                navigator.mediaSession.setActionHandler('nexttrack', () => {
                    nextBtn.click();
                });
            }

            // Funkcje pomocnicze
            function formatTime(seconds) {
                const minutes = Math.floor(seconds / 60);
                const secs = Math.floor(seconds % 60);
                return `${minutes}:${secs < 10 ? '0' : ''}${secs}`;
            }

            function updateNowPlaying(songName) {
                nowPlaying.textContent = `Teraz odtwarzane: ${songName}`;
                
                // Aktualizacja MediaSession
                if ('mediaSession' in navigator) {
                    navigator.mediaSession.metadata = new MediaMetadata({
                        title: songName,
                        artist: 'Odtwarzacz MP3',
                        artwork: [
                            { src: 'icon-192x192.png', sizes: '192x192', type: 'image/png' },
                            { src: 'icon-512x512.png', sizes: '512x512', type: 'image/png' }
                        ]
                    });
                }
            }

            function highlightCurrentSong() {
                const song = currentPlaylist[currentSongIndex];
                if (!song) return;

                const allItems = fileList.querySelectorAll('li');
                allItems.forEach(item => {
                    item.classList.remove('playing');
                    if (item.getAttribute('data-path') === song.path) {
                        item.classList.add('playing');
                        item.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
                    }
                });
            }

            // Obsługa odtwarzania
            function playCurrentSong() {
                if (currentSongIndex === -1 || currentPlaylist.length === 0) return;
                
                const song = currentPlaylist[currentSongIndex];
                audioPlayer.src = `${baseDir}/${song.path}`;
                
                audioPlayer.play()
                    .then(() => {
                        initAudioContext();
                        drawOscilloscope();
                        updateNowPlaying(song.name);
                        highlightCurrentSong();
                        
                        // Włącz odtwarzanie w tle na iOS
                        if (typeof window.AudioContext !== 'undefined' || 
                            typeof window.webkitAudioContext !== 'undefined') {
                            audioContext.resume();
                        }
                    })
                    .catch(error => console.error('Błąd odtwarzania:', error));
            }

            // Obsługa przycisków
            playBtn.addEventListener('click', () => {
                if (audioPlayer.paused) {
                    if (currentPlaylist.length === 0 && allSongs.length > 0) {
                        currentPlaylist = [...allSongs];
                        currentSongIndex = 0;
                    }
                    if (currentPlaylist.length > 0) {
                        if (audioPlayer.currentTime === 0 || audioPlayer.ended) {
                            playCurrentSong();
                        } else {
                            audioPlayer.play();
                        }
                    }
                }
            });

            pauseBtn.addEventListener('click', () => {
                if (!audioPlayer.paused) {
                    audioPlayer.pause();
                }
            });

            stopBtn.addEventListener('click', () => {
                audioPlayer.pause();
                audioPlayer.currentTime = 0;
                progressBar.value = 0;
                currentTimeDisplay.textContent = '0:00';
            });

            prevBtn.addEventListener('click', () => {
                if (currentPlaylist.length === 0) return;
                currentSongIndex = (currentSongIndex - 1 + currentPlaylist.length) % currentPlaylist.length;
                playCurrentSong();
            });

            nextBtn.addEventListener('click', () => {
                if (currentPlaylist.length === 0) return;
                currentSongIndex = (currentSongIndex + 1) % currentPlaylist.length;
                playCurrentSong();
            });

            // Regulacja głośności
            volumeControl.addEventListener('input', (e) => {
                audioPlayer.volume = e.target.value;
            });

            // Oscyloskop
            function resizeCanvas() {
                canvas.width = canvas.offsetWidth;
                canvas.height = canvas.offsetHeight;
            }
            resizeCanvas();
            window.addEventListener('resize', resizeCanvas);

            function drawOscilloscope() {
                if (!analyser || audioPlayer.paused) {
                    requestAnimationFrame(drawOscilloscope);
                    return;
                }

                const bufferLength = analyser.frequencyBinCount;
                const dataArray = new Uint8Array(bufferLength);
                analyser.getByteTimeDomainData(dataArray);

                ctx.fillStyle = 'rgb(34, 34, 34)';
                ctx.fillRect(0, 0, canvas.width, canvas.height);

                ctx.lineWidth = 2;
                ctx.strokeStyle = 'rgb(144, 238, 144)';
                ctx.beginPath();

                const sliceWidth = canvas.width / bufferLength;
                let x = 0;

                for (let i = 0; i < bufferLength; i++) {
                    const v = dataArray[i] / 128.0;
                    const y = v * canvas.height / 2;

                    if (i === 0) ctx.moveTo(x, y);
                    else ctx.lineTo(x, y);

                    x += sliceWidth;
                }

                ctx.lineTo(canvas.width, canvas.height/2);
                ctx.stroke();
                requestAnimationFrame(drawOscilloscope);
            }

            // Ładowanie listy plików
            function loadDirectory(path) {
                fetch(`get_directory.php?path=${encodeURIComponent(path)}`)
                    .then(response => {
                        if (!response.ok) throw new Error('Błąd sieci');
                        return response.json();
                    })
                    .then(data => {
                        if (!data.folders || !data.files) throw new Error('Nieprawidłowy format danych');
                        
                        renderFileList(data, path);
                        allSongs = data.files.map(file => ({
                            path: path ? `${path}/${file}` : file,
                            name: file.replace(/^.*[\\/]/, '')
                        }));
                        currentPlaylist = [];
                        currentSongIndex = -1;
                    })
                    .catch(error => {
                        console.error('Błąd ładowania:', error);
                        fileList.innerHTML = '<li>Błąd ładowania listy plików</li>';
                    });
            }

            function renderFileList(files, currentPath) {
                fileList.innerHTML = '';

                if (currentPath !== '') {
                    const parentLi = document.createElement('li');
                    parentLi.className = 'folder';
                    parentLi.setAttribute('data-path', currentPath.split('/').slice(0, -1).join('/'));
                    
                    const icon = document.createElement('span');
                    icon.className = 'file-icon';
                    icon.textContent = '📁';
                    
                    parentLi.appendChild(icon);
                    parentLi.appendChild(document.createTextNode('..'));
                    fileList.appendChild(parentLi);
                }

                files.folders.forEach(folder => {
                    const li = document.createElement('li');
                    li.className = 'folder';
                    li.setAttribute('data-path', currentPath ? `${currentPath}/${folder}` : folder);
                    
                    const icon = document.createElement('span');
                    icon.className = 'file-icon';
                    icon.textContent = '📁';
                    
                    li.appendChild(icon);
                    li.appendChild(document.createTextNode(folder));
                    fileList.appendChild(li);
                });

                files.files.forEach(file => {
                    const li = document.createElement('li');
                    li.className = 'file';
                    li.setAttribute('data-path', currentPath ? `${currentPath}/${file}` : file);
                    
                    const icon = document.createElement('span');
                    icon.className = 'file-icon';
                    icon.textContent = '🎵';
                    
                    li.appendChild(icon);
                    li.appendChild(document.createTextNode(file.replace(/^.*[\\/]/, '')));
                    fileList.appendChild(li);
                });

                fileList.querySelectorAll('li').forEach(li => {
                    li.addEventListener('click', function() {
                        const path = this.getAttribute('data-path');
                        const isFolder = this.classList.contains('folder');
                        
                        if (isFolder) {
                            loadDirectory(path);
                        } else {
                            currentPlaylist = [...allSongs];
                            currentSongIndex = allSongs.findIndex(song => song.path === path);
                            playCurrentSong();
                            nowPlaying.textContent = `Teraz odtwarzane: ${this.textContent.replace(/[📁🎵]/g, '').trim()}`;
                        }
                    });
                });
            }

            // Obsługa zdarzeń audio
            audioPlayer.addEventListener('timeupdate', () => {
                if (audioPlayer.duration) {
                    const progress = (audioPlayer.currentTime / audioPlayer.duration) * 100;
                    progressBar.value = progress;
                    currentTimeDisplay.textContent = formatTime(audioPlayer.currentTime);
                }
            });

            audioPlayer.addEventListener('loadedmetadata', () => {
                durationDisplay.textContent = formatTime(audioPlayer.duration);
            });

            audioPlayer.addEventListener('ended', () => {
                if (currentPlaylist.length > 0) {
                    currentSongIndex = (currentSongIndex + 1) % currentPlaylist.length;
                    playCurrentSong();
                }
            });

            // Inicjalizacja
            loadDirectory('');
            audioPlayer.volume = volumeControl.value;

            // Blokada menu kontekstowego
            document.addEventListener('contextmenu', function(e) {
                e.preventDefault();
            });

            // Blokada klawiszy
            document.addEventListener('keydown', (e) => {
                if (e.key === 'F12' || (e.ctrlKey && e.shiftKey && (e.key === 'I' || e.key === 'J')) || (e.ctrlKey && e.key === 'U')) {
                    e.preventDefault();
                }
            });

            // Obsługa trybu PWA
            window.addEventListener('beforeinstallprompt', (e) => {
                e.preventDefault();
            });
        });
    </script>
</body>
</html>